//////////////////////////////////////////////////////////////////////////////
//
//  Copyright 2013 by Autodesk, Inc.  All rights reserved.
//
// This computer source code and related instructions and comments
// are the unpublished confidential and proprietary information of
// Autodesk, Inc. and are protected under applicable copyright and
// trade secret law.  They may not be disclosed to, copied or used
// by any third party without the prior written consent of Autodesk, Inc.
//
//////////////////////////////////////////////////////////////////////////////

$.Controller('MarkerLocationController',
    {
        InputState: { ReadOnlyState: 0, EdittingState: 1 },
        location: { longitude: null, latitude: null, elevation: 0.0 },
        existingCurrentCSId: null,
        currentCSId: null,
        //static methods
        executeNativeSync: function (param) {

            return exec(JSON.stringify(param));
        },
        enableNextButton: function () {
            var s = $('input.invalid');
              if (s.length > 0)
                $("#buttonsRow").data("controllers")['buttons_row'].setNextButtonEnabled(false);
            else
                $("#buttonsRow").data("controllers")['buttons_row'].setNextButtonEnabled(true);
        },
        disableNextButton: function () {
            $("#buttonsRow").data("controllers")['buttons_row'].setNextButtonEnabled(false);
        },
        disablePanthere: function (disable) {
            if (IS_ONLINE_MODE) {
                $('#panthere').attr("disabled", disable);
                if(disable)
                    $('#panthere').attr("tabIndex", -1);
                else
                    $('#panthere').removeAttr("tabIndex");
            } else {
                //offline mode - force to disable
                $('#panthere').attr("disabled", true);
                $('#panthere').attr("tabIndex", -1);
            }
        },
        getAngleFormat: function (angle) {
            if (typeof angle === 'undefined' || angle == null || angle.toString().trim().length == 0) return angle;
            var res = MarkerLocationController.validateAngleFormat(angle, -360, 360);
            if (res.isValid)
                return res.degreevaluegeolatlongformat.toString();
            else return angle.toString();
        },
        getFloatFormat: function (floatStr) {
            if (typeof angle === 'undefined' || angle == null || angle.toString().trim().length == 0) return angle;
            var res = MarkerLocationController.validateAngleFormat(angle, -360, 360);
            if (res.isValid)
                return res.floatvalue.toFixed(LUPREC);
            else return floatStr;
        },
        validateAngleFormat: function (angle, minimumValueInDegree, maximumValueInDegree) {

            if (typeof angle === 'undefined' || angle == null || angle.length == 0 || angle == $.localize("ID_NOT_DEFINED", SELECTED_LANG))
                return { isValid: false };

            var args = { 'functionName': 'Ac_Geo.convertAngleStringToFloat',
                'invokeAsCommand': false,
                'functionParams': {
                    'angle': angle.toString(),
                    'maxmiumInDegree': maximumValueInDegree,
                    'minimumInDegree': minimumValueInDegree
                }
            };
            var jsonStr = MarkerLocationController.executeNativeSync(args);
            jsonStr = jsonStr.replace("'", "\'");
            var res = JSON.parse(jsonStr);

            var funcRes = {
                isValid: res.isvalid,
                degreevalue: res.degreevalue,
                degreevaluegeolatlongformat: res.degreevaluegeolatlongformat
            };
            return funcRes;
        },
        validateFloatFormat: function(floatStr) {
            if (typeof floatStr === 'undefined' || floatStr == null || floatStr.length == 0 || floatStr == $.localize("ID_NOT_DEFINED", SELECTED_LANG))
                return { isValid: false };

            var args = { 'functionName': 'Ac_Geo.convertFloatStringToFloat',
                'invokeAsCommand': false,
                'functionParams': {
                    'floatstr': floatStr.toString(),
                }
            };
            var jsonStr = MarkerLocationController.executeNativeSync(args);
            jsonStr = jsonStr.replace("'", "\'");
            var res = JSON.parse(jsonStr);

            var funcRes = {
                isValid: res.isvalid,
                floatvalue: res.floatvalue,
            };
            return funcRes;
        },
        geoLocationMapDlgInvokeTermsLink: function () {
            var args = JSON.stringify({ 'functionName': 'Ac_Geo.geoLocationMapDlgInvokeTermsLink', 'invokeAsCommand': false });
            execAsync(args, function (result) { }, function (result) { });
        },
        hideTermsHyperlinks: function () {
            $("#termsContainer").hide();
        },

        // Meant for handling keydown input events. Returns true if event was handled
        handleKeyDownEvent: function(event) {
            // When ESC within lat/long/elevation edit boxes, do not close the dialog!
            // When ENTER within lat/long/elevation edit boxes, we commit the values.
            if ($("#lat").is(":focus") ||
                $("#long").is(":focus") ||
                $("#elevation").is(":focus")) {
                if (event.keyCode == 13) { // ENTER
                    $('#markerInfoAndListCS').marker_location('validateAcceptInputs');
                    return true;
                }
            }
            return false;
        }
    },
//instance methods
    {
    init: function (element, options) {
        this.mapServiceHandler = options.mapServiceHandler;
        this.markerPin = null;
        this.locationChanged = false;

        this.element.html('marker_info_template', {});
        this.setInputState(MarkerLocationController.InputState.ReadOnlyState);

        $('label').marktext();
        $('#elevation').val((0.0).toFixed(LUPREC)).change();
        $('#latLabel').html($.localize("ID_LATITUDE", SELECTED_LANG));
        $('#longLabel').html($.localize("ID_LONGITUDE", SELECTED_LANG));
        $('#elevationLabel').html($.localize("ID_ELEVATION", SELECTED_LANG));
        $('#metersLabel').html($.localize("ID_METERS", SELECTED_LANG));
        $('#termsContainer').html($.localize("ID_TERMS", SELECTED_LANG));

        MarkerLocationController.disablePanthere(true);

        $('#buttonsRow').buttons_row({ mapService: this.mapServiceHandler });

        MarkerLocationController.location = { longitude: null, latitude: null, elevation: 0.0 };
        //populate existing coordinate system and long/lat values
        if (globalQueries.hasOwnProperty('lat') && globalQueries.hasOwnProperty('long')) {
            MarkerLocationController.location.latitude = parseFloat(globalQueries["lat"]);
            MarkerLocationController.location.longitude = parseFloat(globalQueries["long"]);
        }
        if (globalQueries.hasOwnProperty('elevation')) {
            MarkerLocationController.location.elevation = parseFloat(globalQueries["elevation"]);
            $('#elevation').val(MarkerLocationController.location.elevation).change();
        }
        MarkerLocationController.existingCurrentCSId = null;
        if (globalQueries.hasOwnProperty('curCSId')) {
            MarkerLocationController.existingCurrentCSId = globalQueries['curCSId'];
        }
        if (MarkerLocationController.location.latitude != null) {
            this.dropMarker(MarkerLocationController.location);
            this.locationChanged = false; // dropMarker will set it to true
        }

        // Enable "Continue..." button if data is populated from KMLKMZ file
        if (globalQueries.hasOwnProperty('fromKML')) {
            if (globalQueries['fromKML'] > 0)
                MarkerLocationController.enableNextButton();
        }

        if (!IS_ONLINE_MODE) {
            this.setInputState(MarkerLocationController.InputState.EdittingState);
            MarkerLocationController.hideTermsHyperlinks();
        }
        else {
            // we don't want to show the access key for lat/long labels if online,
            // since the corresponding edit boxes are disabled.
            $('#latLabel span').removeClass('accesskey');
            $('#longLabel span').removeClass('accesskey');
        }
        
        CAN_CANCEL = true;
    },
    destroy: function () {
        if (this.mapServiceHandler) {
            if (this.markerPinDragEndHandler) {
                this.mapServiceHandler.removeHandler(this.markerPinDragEndHandler);
            }
            this.mapServiceHandler.removeEntity(this.markerPin);
        }
        //destroy button row
        if ($('#buttonsRow').controllers().length != 0) {
            $('#buttonsRow').buttons_row('destroy');
        }
        $('#buttonsRow').html("");
        this._super();
    },
    panThere: function () {
        if (this.mapServiceHandler) {
            this.mapServiceHandler.panMapTo(this.mapServiceHandler.getPinLocation(this.markerPin));
        }
    },
    dropMarkerWithoutGetCoordinateSystem: function (location) {
        if (!location) return;
        MarkerLocationController.location = location;

        if (this.mapServiceHandler) {
            if (!this.markerPin) {
                this.markerPin = this.mapServiceHandler.addMarker(location);
                this.markerPinDragEndHandler = this.mapServiceHandler.addHandler(this.markerPin, "dragend", this.updateLocationAfterDrag(this));
            } else {
                this.mapServiceHandler.setPinLocation(this.markerPin, location)
            }
        }
    },
    dropMarker: function (location, fromLatLongElevInputs) {
        if (!location) return;
        if (!this.markerPin) {
            this.setLocation(location, fromLatLongElevInputs)
            if (this.mapServiceHandler) {
                this.markerPin = this.mapServiceHandler.addMarker(location);
                this.markerPinDragEndHandler = this.mapServiceHandler.addHandler(this.markerPin, "dragend", this.updateLocationAfterDrag(this));
            }
        } else {
            this.mapServiceHandler.setPinLocation(this.markerPin, location)
            this.setLocation(location, fromLatLongElevInputs);
        }
        
        this.locationChanged = true;
    },
    updateLocationAfterDrag: function (self) {
        return function (event) {
            if (event) {
                var pinLoc = self.mapServiceHandler.getPinLocation(self.markerPin);
                self.setLocation(pinLoc);
                MarkerLocationController.enableNextButton();
            }
        }
    },
    setLocation: function (location, fromLatLongElevInputs) {       
        var self = this;
        setTimeout(function () {

            MarkerLocationController.location.longitude = location.longitude;
            MarkerLocationController.location.latitude = location.latitude;

            if(!fromLatLongElevInputs) {
                $('#lat').val(MarkerLocationController.getAngleFormat(location.latitude)).change();
                $('#long').val(MarkerLocationController.getAngleFormat(location.longitude)).change();
            }

            if (fromLatLongElevInputs && !(typeof location.elevation === 'undefined') && location.elevation != null && !isNaN(location.elevation)) {            
                MarkerLocationController.location.elevation = location.elevation;
            }            

            MarkerLocationController.disablePanthere(false);
            MarkerLocationController.enableNextButton();
            $('#elevation').attr("disabled", false);            

        }, 200);
    },
    setInputState: function (state) {
        if (state == MarkerLocationController.InputState.ReadOnlyState) {
            $('#lat').attr("disabled", true);
            $('#long').attr("disabled", true);
            $('#elevation').attr("disabled", true);

            // in online mode, the lat/long/elevation are supposed to be always non-editable
            // and hence supposed to look like labels.
            $('#latValue').addClass('latLongElevDisabled');
            $('#longValue').addClass('latLongElevDisabled');

        } else if (state == MarkerLocationController.InputState.EdittingState) {
            $('#lat').attr("disabled", false);
            $('#lat').focus();
            $('#long').attr("disabled", false);
            $('#elevation').attr("disabled", false);

            $('#latValue').removeClass('latLongElevDisabled');
            $('#longValue').removeClass('latLongElevDisabled');

            //save button is disabled too
            MarkerLocationController.disableNextButton();
        }
        $('#lat').removeClass("invalid");
        $('#long').removeClass("invalid");
        $('#elevation').removeClass("invalid");

    },
    '#lat keypress': function (el, ev) {

    },
    '#long keypress': function (el, ev) {

    },

    validateLat: function () {
        var latStr = $('#lat').val();
        
        // validate only if string contains non-whitespace
        if(/\S/.test(latStr))
        {
            var resLat = MarkerLocationController.validateAngleFormat(latStr, -90.0, 90.0);
            if (resLat.isValid) {
                $('#lat').removeClass('invalid');
                if (!(typeof resLat.degreevaluegeolatlongformat === 'undefined')) {
                    $('#lat').val(resLat.degreevaluegeolatlongformat.toString());
                }
            }
            else {
                $('#lat').addClass('invalid');
            }
            return resLat;
        }
        
        return {'isValid':0,'degreevalue':0};
    },
    validateLong: function () {
        var longStr = $('#long').val();
        
        // validate only if string contains non-whitespace
        if(/\S/.test(longStr))
        {
            var resLong = MarkerLocationController.validateAngleFormat(longStr, -180.0, 180.0);
            if (resLong.isValid) {
                $('#long').removeClass('invalid');
                if (!(typeof resLong.degreevaluegeolatlongformat === 'undefined')) {
                    $('#long').val(resLong.degreevaluegeolatlongformat.toString());
                }
            }
            else {
                $('#long').addClass('invalid');
            }
            return resLong;
        }        

        return {'isValid':0,'degreevalue':0};
    },
    validateElevation: function () {
        var elevStr = $('#elevation').val();
        
        // validate only if string contains non-whitespace
        if(/\S/.test(elevStr))
        {
            var resElevation = MarkerLocationController.validateFloatFormat(elevStr);

            if (resElevation.isValid){
                 $('#elevation').removeClass('invalid');
                 if (!(typeof resElevation.floatvalue === 'undefined')) {                
                    $('#elevation').val(resElevation.floatvalue.toFixed(LUPREC));
                 }
            } else{
                 $('#elevation').addClass('invalid');
            }
            return resElevation;
        }
        else {
            // automatically change empty to zero
            $('#elevation').val((0.0).toFixed(LUPREC)).change();
        }
        
        return {'isValid':0,'degreevalue':0};
    },
    validateInputs: function () {
        var resLat = this.validateLat();
        var resLong = this.validateLong();
        var resElevation = this.validateElevation();
        return {
            isValid: resLat.isValid && resLong.isValid && resElevation.isValid,
            lat: resLat.degreevalue,
            long: resLong.degreevalue,
            elevation: resElevation.floatvalue
        };
    },
    validateAcceptInputs: function () {
        var res = this.validateInputs();
        if (res.isValid) {
            var longitude = res.long;
            var latitude = res.lat;
            var elevation = res.elevation;
            var location = { longitude: longitude, latitude: latitude, elevation: elevation };
            this.dropMarker(location, true);

            MarkerLocationController.enableNextButton();
        } else {
            MarkerLocationController.disableNextButton();
        }
        return false;
    },
    '#lat change': function (el, ev) {
        this.validateAcceptInputs();
    },
    '#long change': function (el, ev) {
        this.validateAcceptInputs();
    },
    '#elevation change': function (el, ev) {
        this.validateAcceptInputs();
    } 
});



// SIG // Begin signature block
// SIG // MIIZNgYJKoZIhvcNAQcCoIIZJzCCGSMCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFCJiCRaXKw3C
// SIG // u5/gcI0ynOnC1PdhoIIUMDCCA+4wggNXoAMCAQICEH6T
// SIG // 6/t8xk5Z6kuad9QG/DswDQYJKoZIhvcNAQEFBQAwgYsx
// SIG // CzAJBgNVBAYTAlpBMRUwEwYDVQQIEwxXZXN0ZXJuIENh
// SIG // cGUxFDASBgNVBAcTC0R1cmJhbnZpbGxlMQ8wDQYDVQQK
// SIG // EwZUaGF3dGUxHTAbBgNVBAsTFFRoYXd0ZSBDZXJ0aWZp
// SIG // Y2F0aW9uMR8wHQYDVQQDExZUaGF3dGUgVGltZXN0YW1w
// SIG // aW5nIENBMB4XDTEyMTIyMTAwMDAwMFoXDTIwMTIzMDIz
// SIG // NTk1OVowXjELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFFN5
// SIG // bWFudGVjIENvcnBvcmF0aW9uMTAwLgYDVQQDEydTeW1h
// SIG // bnRlYyBUaW1lIFN0YW1waW5nIFNlcnZpY2VzIENBIC0g
// SIG // RzIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
// SIG // AQCxrLNJVEuXHBIK2CV5kSJXKm/cuCbEQ3Nrwr8uUFr7
// SIG // FMJ2jkMBJUO0oeJF9Oi3e8N0zCLXtJQAAvdN7b+0t0Qk
// SIG // a81fRTvRRM5DEnMXgotptCvLmR6schsmTXEfsTHd+1Fh
// SIG // AlOmqvVJLAV4RaUvic7nmef+jOJXPz3GktxK+Hsz5HkK
// SIG // +/B1iEGc/8UDUZmq12yfk2mHZSmDhcJgFMTIyTsU2sCB
// SIG // 8B8NdN6SIqvK9/t0fCfm90obf6fDni2uiuqm5qonFn1h
// SIG // 95hxEbziUKFL5V365Q6nLJ+qZSDT2JboyHylTkhE/xni
// SIG // RAeSC9dohIBdanhkRc1gRn5UwRN8xXnxycFxAgMBAAGj
// SIG // gfowgfcwHQYDVR0OBBYEFF+a9W5czMx0mtTdfe8/2+xM
// SIG // gC7dMDIGCCsGAQUFBwEBBCYwJDAiBggrBgEFBQcwAYYW
// SIG // aHR0cDovL29jc3AudGhhd3RlLmNvbTASBgNVHRMBAf8E
// SIG // CDAGAQH/AgEAMD8GA1UdHwQ4MDYwNKAyoDCGLmh0dHA6
// SIG // Ly9jcmwudGhhd3RlLmNvbS9UaGF3dGVUaW1lc3RhbXBp
// SIG // bmdDQS5jcmwwEwYDVR0lBAwwCgYIKwYBBQUHAwgwDgYD
// SIG // VR0PAQH/BAQDAgEGMCgGA1UdEQQhMB+kHTAbMRkwFwYD
// SIG // VQQDExBUaW1lU3RhbXAtMjA0OC0xMA0GCSqGSIb3DQEB
// SIG // BQUAA4GBAAMJm495739ZMKrvaLX64wkdu0+CBl03X6ZS
// SIG // nxaN6hySCURu9W3rWHww6PlpjSNzCxJvR6muORH4KrGb
// SIG // sBrDjutZlgCtzgxNstAxpghcKnr84nodV0yoZRjpeUBi
// SIG // JZZux8c3aoMhCI5B6t3ZVz8dd0mHKhYGXqY4aiISo1EZ
// SIG // g362MIIEozCCA4ugAwIBAgIQDs/0OMj+vzVuBNhqmBsa
// SIG // UDANBgkqhkiG9w0BAQUFADBeMQswCQYDVQQGEwJVUzEd
// SIG // MBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xMDAu
// SIG // BgNVBAMTJ1N5bWFudGVjIFRpbWUgU3RhbXBpbmcgU2Vy
// SIG // dmljZXMgQ0EgLSBHMjAeFw0xMjEwMTgwMDAwMDBaFw0y
// SIG // MDEyMjkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMR0wGwYD
// SIG // VQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlvbjE0MDIGA1UE
// SIG // AxMrU3ltYW50ZWMgVGltZSBTdGFtcGluZyBTZXJ2aWNl
// SIG // cyBTaWduZXIgLSBHNDCCASIwDQYJKoZIhvcNAQEBBQAD
// SIG // ggEPADCCAQoCggEBAKJjCzlEuLsjp0RJuw7/ofBhClOT
// SIG // sJjbrSwPSsVu/4Y8U1UPFc4EPyv9qZaW2b5heQtbyUyG
// SIG // duXgQ0sile7CK0PBn9hotI5AT+6FOLkRxSPyZFjwFTJv
// SIG // TlehroikAtcqHs1L4d1j1ReJMluwXplaqJ0oUA4X7pbb
// SIG // YTtFUR3PElYLkkf8q672Zj1HrHBy55LnX80QucSDZJQZ
// SIG // vSWA4ejSIqXQugJ6oXeTW2XD7hd0vEGGKtwITIySjJEt
// SIG // nndEH2jWqHR32w5bMotWizO92WPISZ06xcXqMwvS8aMb
// SIG // 9Iu+2bNXizveBKd6IrIkri7HcMW+ToMmCPsLvalPmQjh
// SIG // EChyqs0CAwEAAaOCAVcwggFTMAwGA1UdEwEB/wQCMAAw
// SIG // FgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwDgYDVR0PAQH/
// SIG // BAQDAgeAMHMGCCsGAQUFBwEBBGcwZTAqBggrBgEFBQcw
// SIG // AYYeaHR0cDovL3RzLW9jc3Aud3Muc3ltYW50ZWMuY29t
// SIG // MDcGCCsGAQUFBzAChitodHRwOi8vdHMtYWlhLndzLnN5
// SIG // bWFudGVjLmNvbS90c3MtY2EtZzIuY2VyMDwGA1UdHwQ1
// SIG // MDMwMaAvoC2GK2h0dHA6Ly90cy1jcmwud3Muc3ltYW50
// SIG // ZWMuY29tL3Rzcy1jYS1nMi5jcmwwKAYDVR0RBCEwH6Qd
// SIG // MBsxGTAXBgNVBAMTEFRpbWVTdGFtcC0yMDQ4LTIwHQYD
// SIG // VR0OBBYEFEbGaaMOShQe1UzaUmMXP142vA3mMB8GA1Ud
// SIG // IwQYMBaAFF+a9W5czMx0mtTdfe8/2+xMgC7dMA0GCSqG
// SIG // SIb3DQEBBQUAA4IBAQB4O7SRKgBM8I9iMDd4o4QnB28Y
// SIG // st4l3KDUlAOqhk4ln5pAAxzdzuN5yyFoBtq2MrRtv/Qs
// SIG // JmMz5ElkbQ3mw2cO9wWkNWx8iRbG6bLfsundIMZxD82V
// SIG // dNy2XN69Nx9DeOZ4tc0oBCCjqvFLxIgpkQ6A0RH83Vx2
// SIG // bk9eDkVGQW4NsOo4mrE62glxEPwcebSAe6xp9P2ctgwW
// SIG // K/F/Wwk9m1viFsoTgW0ALjgNqCmPLOGy9FqpAa8VnCwv
// SIG // SRvbIrvD/niUUcOGsYKIXfA9tFGheTMrLnu53CAJE3Hr
// SIG // ahlbz+ilMFcsiUk/uc9/yb8+ImhjU5q9aXSsxR08f5Lg
// SIG // w7wc2AR1MIIFhTCCBG2gAwIBAgIQKcFbP6rNUmpOZ708
// SIG // Tn4/8jANBgkqhkiG9w0BAQUFADCBtDELMAkGA1UEBhMC
// SIG // VVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYD
// SIG // VQQLExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYD
// SIG // VQQLEzJUZXJtcyBvZiB1c2UgYXQgaHR0cHM6Ly93d3cu
// SIG // dmVyaXNpZ24uY29tL3JwYSAoYykxMDEuMCwGA1UEAxMl
// SIG // VmVyaVNpZ24gQ2xhc3MgMyBDb2RlIFNpZ25pbmcgMjAx
// SIG // MCBDQTAeFw0xMjA3MjUwMDAwMDBaFw0xNTA5MjAyMzU5
// SIG // NTlaMIHIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKQ2Fs
// SIG // aWZvcm5pYTETMBEGA1UEBxMKU2FuIFJhZmFlbDEWMBQG
// SIG // A1UEChQNQXV0b2Rlc2ssIEluYzE+MDwGA1UECxM1RGln
// SIG // aXRhbCBJRCBDbGFzcyAzIC0gTWljcm9zb2Z0IFNvZnR3
// SIG // YXJlIFZhbGlkYXRpb24gdjIxHzAdBgNVBAsUFkRlc2ln
// SIG // biBTb2x1dGlvbnMgR3JvdXAxFjAUBgNVBAMUDUF1dG9k
// SIG // ZXNrLCBJbmMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQCoYmDrmd0Gq8ezSsDlfgaJFEFplNPNhWzM
// SIG // 2uFQaYAB/ggpQ11+N4B6ao+TqrNIWDIqt3JKhaU889nx
// SIG // l/7teWGwuOurstI2Z0bEDhXiXam/bicK2HVLyntliQ+6
// SIG // tT+nlgfN8tgB2NzM0BpE1YCnU2b6DwQw4V7BV+/F//83
// SIG // yGFOpePlumzXxNw9EKWkaq81slmmTxf7UxZgP9PGbLw8
// SIG // gLAPk4PTJI97+5BBqhkLb1YqSfWn3PNMfsNKhw/VwAN0
// SIG // dRKeM6H8SkOdz+osr+NyH86lsKQuics4fwK5uFSHQHsI
// SIG // t6Z0tqWvminRqceUi9ugRlGryh9X1ZqCqfL/ggdzYa3Z
// SIG // AgMBAAGjggF7MIIBdzAJBgNVHRMEAjAAMA4GA1UdDwEB
// SIG // /wQEAwIHgDBABgNVHR8EOTA3MDWgM6Axhi9odHRwOi8v
// SIG // Y3NjMy0yMDEwLWNybC52ZXJpc2lnbi5jb20vQ1NDMy0y
// SIG // MDEwLmNybDBEBgNVHSAEPTA7MDkGC2CGSAGG+EUBBxcD
// SIG // MCowKAYIKwYBBQUHAgEWHGh0dHBzOi8vd3d3LnZlcmlz
// SIG // aWduLmNvbS9ycGEwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
// SIG // cQYIKwYBBQUHAQEEZTBjMCQGCCsGAQUFBzABhhhodHRw
// SIG // Oi8vb2NzcC52ZXJpc2lnbi5jb20wOwYIKwYBBQUHMAKG
// SIG // L2h0dHA6Ly9jc2MzLTIwMTAtYWlhLnZlcmlzaWduLmNv
// SIG // bS9DU0MzLTIwMTAuY2VyMB8GA1UdIwQYMBaAFM+Zqep7
// SIG // JvRLyY6P1/AFJu/j0qedMBEGCWCGSAGG+EIBAQQEAwIE
// SIG // EDAWBgorBgEEAYI3AgEbBAgwBgEBAAEB/zANBgkqhkiG
// SIG // 9w0BAQUFAAOCAQEA2OkGvuiY7TyI6yVTQAYmTO+MpOFG
// SIG // C8MflHSbofJiuLxrS1KXbkzsAPFPPsU1ouftFhsXFtDQ
// SIG // 8rMTq/jwugTpbJUREV0buEkLl8AKRhYQTKBKg1I/puBv
// SIG // bkJocDE0pRwtBz3xSlXXEwyYPcbCOnrM3OZ5bKx1Qiii
// SIG // vixlcGWhO3ws904ssutPFf4mV5PDi3U2Yp1HgbBK/Um/
// SIG // FLr6YAYeZaA8KY1CfQEisF3UKTwm72d7S+fJf++SOGea
// SIG // K0kumehVcbavQJTOVebuZ9V+qU0nk1lMrqve9BnQK69B
// SIG // QqNZu77vCO0wm81cfynAxkOYKZG3idY47qPJOgXKkwmI
// SIG // 2+92ozCCBgowggTyoAMCAQICEFIA5aolVvwahu2WydRL
// SIG // M8cwDQYJKoZIhvcNAQEFBQAwgcoxCzAJBgNVBAYTAlVT
// SIG // MRcwFQYDVQQKEw5WZXJpU2lnbiwgSW5jLjEfMB0GA1UE
// SIG // CxMWVmVyaVNpZ24gVHJ1c3QgTmV0d29yazE6MDgGA1UE
// SIG // CxMxKGMpIDIwMDYgVmVyaVNpZ24sIEluYy4gLSBGb3Ig
// SIG // YXV0aG9yaXplZCB1c2Ugb25seTFFMEMGA1UEAxM8VmVy
// SIG // aVNpZ24gQ2xhc3MgMyBQdWJsaWMgUHJpbWFyeSBDZXJ0
// SIG // aWZpY2F0aW9uIEF1dGhvcml0eSAtIEc1MB4XDTEwMDIw
// SIG // ODAwMDAwMFoXDTIwMDIwNzIzNTk1OVowgbQxCzAJBgNV
// SIG // BAYTAlVTMRcwFQYDVQQKEw5WZXJpU2lnbiwgSW5jLjEf
// SIG // MB0GA1UECxMWVmVyaVNpZ24gVHJ1c3QgTmV0d29yazE7
// SIG // MDkGA1UECxMyVGVybXMgb2YgdXNlIGF0IGh0dHBzOi8v
// SIG // d3d3LnZlcmlzaWduLmNvbS9ycGEgKGMpMTAxLjAsBgNV
// SIG // BAMTJVZlcmlTaWduIENsYXNzIDMgQ29kZSBTaWduaW5n
// SIG // IDIwMTAgQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQD1I0tepdeKuzLp1Ff37+THJn6tGZj+qJ19
// SIG // lPY2axDXdYEwfwRof8srdR7NHQiM32mUpzejnHuA4Jnh
// SIG // 7jdNX847FO6G1ND1JzW8JQs4p4xjnRejCKWrsPvNamKC
// SIG // TNUh2hvZ8eOEO4oqT4VbkAFPyad2EH8nA3y+rn59wd35
// SIG // BbwbSJxp58CkPDxBAD7fluXF5JRx1lUBxwAmSkA8taEm
// SIG // qQynbYCOkCV7z78/HOsvlvrlh3fGtVayejtUMFMb32I0
// SIG // /x7R9FqTKIXlTBdOflv9pJOZf9/N76R17+8V9kfn+Bly
// SIG // 2C40Gqa0p0x+vbtPDD1X8TDWpjaO1oB21xkupc1+NC2J
// SIG // AgMBAAGjggH+MIIB+jASBgNVHRMBAf8ECDAGAQH/AgEA
// SIG // MHAGA1UdIARpMGcwZQYLYIZIAYb4RQEHFwMwVjAoBggr
// SIG // BgEFBQcCARYcaHR0cHM6Ly93d3cudmVyaXNpZ24uY29t
// SIG // L2NwczAqBggrBgEFBQcCAjAeGhxodHRwczovL3d3dy52
// SIG // ZXJpc2lnbi5jb20vcnBhMA4GA1UdDwEB/wQEAwIBBjBt
// SIG // BggrBgEFBQcBDARhMF+hXaBbMFkwVzBVFglpbWFnZS9n
// SIG // aWYwITAfMAcGBSsOAwIaBBSP5dMahqyNjmvDz4Bq1EgY
// SIG // LHsZLjAlFiNodHRwOi8vbG9nby52ZXJpc2lnbi5jb20v
// SIG // dnNsb2dvLmdpZjA0BgNVHR8ELTArMCmgJ6AlhiNodHRw
// SIG // Oi8vY3JsLnZlcmlzaWduLmNvbS9wY2EzLWc1LmNybDA0
// SIG // BggrBgEFBQcBAQQoMCYwJAYIKwYBBQUHMAGGGGh0dHA6
// SIG // Ly9vY3NwLnZlcmlzaWduLmNvbTAdBgNVHSUEFjAUBggr
// SIG // BgEFBQcDAgYIKwYBBQUHAwMwKAYDVR0RBCEwH6QdMBsx
// SIG // GTAXBgNVBAMTEFZlcmlTaWduTVBLSS0yLTgwHQYDVR0O
// SIG // BBYEFM+Zqep7JvRLyY6P1/AFJu/j0qedMB8GA1UdIwQY
// SIG // MBaAFH/TZafC3ey78DAJ80M5+gKvMzEzMA0GCSqGSIb3
// SIG // DQEBBQUAA4IBAQBWIuY0pMRhy0i5Aa1WqGQP2YyRxLvM
// SIG // DOWteqAif99HOEotbNF/cRp87HCpsfBP5A8MU/oVXv50
// SIG // mEkkhYEmHJEUR7BMY4y7oTTUxkXoDYUmcwPQqYxkbdxx
// SIG // kuZFBWAVWVE5/FgUa/7UpO15awgMQXLnNyIGCb4j6T9E
// SIG // mh7pYZ3MsZBc/D3SjaxCPWU21LQ9QCiPmxDPIybMSyDL
// SIG // kB9djEw0yjzY5TfWb6UgvTTrJtmuDefFmvehtCGRM2+G
// SIG // 6Fi7JXx0Dlj+dRtjP84xfJuPG5aexVN2hFucrZH6rO2T
// SIG // ul3IIVPCglNjrxINUIcRGz1UUpaKLJw9khoImgUux5Ol
// SIG // SJHTMYIEcjCCBG4CAQEwgckwgbQxCzAJBgNVBAYTAlVT
// SIG // MRcwFQYDVQQKEw5WZXJpU2lnbiwgSW5jLjEfMB0GA1UE
// SIG // CxMWVmVyaVNpZ24gVHJ1c3QgTmV0d29yazE7MDkGA1UE
// SIG // CxMyVGVybXMgb2YgdXNlIGF0IGh0dHBzOi8vd3d3LnZl
// SIG // cmlzaWduLmNvbS9ycGEgKGMpMTAxLjAsBgNVBAMTJVZl
// SIG // cmlTaWduIENsYXNzIDMgQ29kZSBTaWduaW5nIDIwMTAg
// SIG // Q0ECECnBWz+qzVJqTme9PE5+P/IwCQYFKw4DAhoFAKBw
// SIG // MBAGCisGAQQBgjcCAQwxAjAAMBkGCSqGSIb3DQEJAzEM
// SIG // BgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgor
// SIG // BgEEAYI3AgEVMCMGCSqGSIb3DQEJBDEWBBSh/Bq3tAue
// SIG // 2t4Bch1pZDSJP8SD2zANBgkqhkiG9w0BAQEFAASCAQBf
// SIG // ve7ZcPicXxIaXtAjsGCfOJd4iQCGvQwhkFi3q5kYPmbL
// SIG // HB3Gb8Qz1VZvtiXosiAhWgyB3BTd3IknvXBEoPyww57h
// SIG // vyXRT8VWdjsixMZxJ2iIOv81Rkdci5Pf5qkm9jci9fq9
// SIG // C3UXEsOz3ns2vjFlnhRql1dDE79AH/J56OnpZLIOKkq6
// SIG // 7GfPU4PH7eeNVQN6W1BnpbXXq+2gQkD97hRGjvMibm3R
// SIG // okhElucyxdABaNexlRbzKxWyWpZzhoRea9DQ/nzKD79E
// SIG // g9b0hF095UZt5O8/LkfMFjuiEd3dBOpaqb2S2RN/b2Mv
// SIG // Ffp9GfsZtt8PSIxjCQ8Ie7hCo5vFrf9EoYICCzCCAgcG
// SIG // CSqGSIb3DQEJBjGCAfgwggH0AgEBMHIwXjELMAkGA1UE
// SIG // BhMCVVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0
// SIG // aW9uMTAwLgYDVQQDEydTeW1hbnRlYyBUaW1lIFN0YW1w
// SIG // aW5nIFNlcnZpY2VzIENBIC0gRzICEA7P9DjI/r81bgTY
// SIG // apgbGlAwCQYFKw4DAhoFAKBdMBgGCSqGSIb3DQEJAzEL
// SIG // BgkqhkiG9w0BBwEwHAYJKoZIhvcNAQkFMQ8XDTE0MDIw
// SIG // NzExMTQxMFowIwYJKoZIhvcNAQkEMRYEFLjlT8MvDe7g
// SIG // UOMzuCtUW/JkZvm5MA0GCSqGSIb3DQEBAQUABIIBAHqf
// SIG // LU7JH4zHy4efbmkW/j6IGYQ2o3gCcE5v4ZFv9axmr8Qz
// SIG // 5Dk8eSXUE7Wc7v4nZrt9nJizowC+PsFbB1wRfry4kXBk
// SIG // bm82Kd+yUYRL/XM+1Zuo6cjyAfv1BdZRiD1G0GrWCoNj
// SIG // RSlj45NeZoZ0nk2MHo2NQlP6Cdrafw9fSSSGEtWhMFtJ
// SIG // iM2pFQ6YPYkMbBCpfCWFteZ+u2Mj2vZ7ebgDdnExHIzP
// SIG // Xzw2I6O8lJ9mv6Ymyp9O9AJZUlAOKspaFvtHNbklvjuc
// SIG // GOJ0SyQoUISMG/UXQFOP+oPDGJr0uHvvdhiowgtbPXa2
// SIG // 72CX2IourcU/f3VAFCk60oQbYHKrwHA=
// SIG // End signature block
